// This is a part of the Microsoft Foundation Classes C++ library.
// Copyright (C) 1992-1999 Microsoft Corporation
// All rights reserved.
//
// This source code is only intended as a supplement to the
// Microsoft Foundation Classes Reference and related
// electronic documentation provided with the library.
// See these sources for detailed information regarding the
// Microsoft Foundation Classes product.

#include "stdafx.h"
#include <commctrl.h>


#ifdef _DEBUG
#undef THIS_FILE
static char THIS_FILE[] = __FILE__;
#endif

#define new DEBUG_NEW

#ifdef AFX_CORE3_SEG
#pragma code_seg(AFX_CORE3_SEG)
#endif

#define TBSTYLE_NO_DROPDOWN_ARROW 0x0080

#if !defined(_WIN32_WCE_NO_CONTROLBARS)

/////////////////////////////////////////////////////////////////////////////
// Backward compatibility on CFrameWnd members

// static
CCeCommandBar* CCeCommandBar::GetDefaultCommandBar(CFrameWnd *pFrameWnd, BOOL bAutoCreateMenu /* = TRUE */)
{
	ASSERT(pFrameWnd != NULL);

	BOOL bControlBarExists = FALSE;

	// Return the first CCeCommandBar (if one exists)
	POSITION pos = pFrameWnd->m_listControlBars.GetHeadPosition();
	while(pos != NULL)
	{
		bControlBarExists = TRUE;
		CControlBar* pBar = (CControlBar*)pFrameWnd->m_listControlBars.GetNext(pos);
		ASSERT(pBar != NULL);
		if(pBar->IsKindOf(RUNTIME_CLASS(CCeCommandBar)))
			return (CCeCommandBar*)pBar; 
	}

	if(bControlBarExists)
	{
		// There are other control bars owned by the CFrameWnd, but no CCeCommandBar.
		// Don't create a CCeCommandBar on the fly.
		ASSERT(FALSE);
		return NULL;
	}

	// The gist of the backward compatibility CFrameWnd functions is the on-the-fly 
	// creation of CCeCommandBars.
	CCeCommandBar* pBar = new CCeCommandBar;
	if(pBar == NULL)
		return NULL;

	if(!pBar->Create(pFrameWnd))
	{
		delete pBar;
		return NULL;
	}

	if(bAutoCreateMenu && (pFrameWnd->m_lpszMenuName != NULL))
	{
		pBar->InsertMenuBar(pFrameWnd->m_lpszMenuName);
#if !(defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300))
		pBar->InsertSeparator(6);
#endif // _WIN32_WCE_PSPC
	}

	pFrameWnd->m_hCommandBar = pBar->m_hWnd;
	pFrameWnd->m_poCommandBar = pFrameWnd; // because CFrameWnd has the old-named functions

	// Allow the CCeCommandBar to delete itself when destroyed.
	pBar->m_bAutoDelete = TRUE;
	
	return pBar;
}

BOOL CFrameWnd::InitCommandBar(LPCTSTR lpszMenuName)
{
	HWND hWndCB;

	if((hWndCB = ResetCommandBar()) == NULL)
		return FALSE;

	CCeCommandBar* pCB = (CCeCommandBar*)(CWnd::FromHandlePermanent(hWndCB));
	ASSERT(pCB != NULL);

	CMenu* pMenu = pCB->InsertMenuBar(lpszMenuName);
	if(pMenu == NULL)
		return FALSE;

	SetMenu(pMenu);
	return TRUE;
}

HWND CFrameWnd::InsertComboBox(int nWidth)
{
	// Note: no CComboBox is generated by this method
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		ASSERT(pBar->m_hBCComboBox == NULL); // Can't add more than one combo box with this 
		                                     // backward compatibility function.
		pBar->InsertComboBox(nWidth, 0xFFFE); // 0xFFFE marks the BC combo box
		return pBar->m_hBCComboBox;
	}
	return NULL;
}

BOOL CFrameWnd::InsertButtons(TBBUTTON arTBButtons[], UINT uiNButtons, int iToolbarBitmapID, int iNImages)
{
	ASSERT(arTBButtons);
	ASSERT(uiNButtons > 0);
	ASSERT(iToolbarBitmapID >= 0);
	ASSERT(iNImages > 0);
	
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);

	if(pBar != NULL)
	{
		if(::CommandBar_AddBitmap(pBar->m_hWnd, AfxGetInstanceHandle(), 
			                      iToolbarBitmapID, iNImages, 0, 0) >= 0)	     
		{
			return pBar->GetToolBarCtrl().AddButtons(uiNButtons, arTBButtons);
		}
	}
	return FALSE;
}

BOOL CFrameWnd::AddBitmap(int iToolbarBitmapID, int iNImages)
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		return pBar->GetToolBarCtrl().AddBitmap(iToolbarBitmapID, iNImages);
	}
	return FALSE;
}

HMENU CFrameWnd::InsertMenu(LPCTSTR lpszMenuName)
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this, FALSE);
	if(pBar != NULL)
	{
		CMenu* pMenu = pBar->InsertMenuBar(lpszMenuName);
		if(pMenu != NULL)
		{
#if !(defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300))
			pBar->InsertSeparator(6);
#endif // _WIN32_WCE_PSPC
			SetMenu(pMenu);
			m_lpszMenuName = lpszMenuName;

			// Detach the menu, because in the past it was detached and 
			// users could attach it to their own CMenu.
			HMENU hMenu = pMenu->Detach();
			return hMenu;
		}
	}
	return NULL;
}

HWND CFrameWnd::ResetCommandBar()
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this, FALSE);
	if(pBar != NULL)
	{
		pBar->ResetCommandBar();
		return pBar->m_hWnd;
	}
	return NULL;
}

BOOL CFrameWnd::AddComboBoxString(UINT uiStringResourceID, long lItemData)
{
	return AddComboBoxString((LPCTSTR)CString(uiStringResourceID), lItemData);
}

BOOL CFrameWnd::AddComboBoxString( LPCTSTR szString, long lItemData )
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		ASSERT((pBar->m_hBCComboBox != NULL) && ::IsWindow(pBar->m_hBCComboBox)); 
		DWORD dwIndex = ::SendMessage(pBar->m_hBCComboBox, CB_ADDSTRING, 0, 
			(LPARAM)(LPCSTR)szString); 
		if(dwIndex >= 0)
		{
			::SendMessage(pBar->m_hBCComboBox, CB_SETITEMDATA, dwIndex, lItemData);
			return TRUE;
		}
	}
	return FALSE;
}

int CFrameWnd::GetComboCount()
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		ASSERT((pBar->m_hBCComboBox != NULL) && ::IsWindow(pBar->m_hBCComboBox)); 
		return (int)::SendMessage(pBar->m_hBCComboBox, CB_GETCOUNT, 0, 0);
	}
	return -1;
}

int CFrameWnd::GetComboCurSel()
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		ASSERT((pBar->m_hBCComboBox != NULL) && ::IsWindow(pBar->m_hBCComboBox)); 
		return (int)::SendMessage(pBar->m_hBCComboBox, CB_GETCURSEL, 0, 0);
	}
	return -1;
}

int CFrameWnd::SetComboCurSel(int nSelect)
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		ASSERT((pBar->m_hBCComboBox != NULL) && ::IsWindow(pBar->m_hBCComboBox)); 
		return (int)::SendMessage(pBar->m_hBCComboBox, CB_SETCURSEL, nSelect, 0);
	}
	return -1;
}

BOOL CFrameWnd::AddAdornments(DWORD dwFlags)
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		return pBar->AddAdornments(dwFlags);
	}
	return FALSE;
}

int CFrameWnd::GetHeight()
{
	CCeCommandBar* pBar = CCeCommandBar::GetDefaultCommandBar(this);
	if(pBar != NULL)
	{
		CRect rect;
		pBar->GetClientRect(&rect);
		return rect.Height();
	}
	return 0;
}

/////////////////////////////////////////////////////////////////////////////
// CCeCommandBar

CCeCommandBar::CCeCommandBar()
{
	m_pMenuArray.SetSize(0);
	m_pComboBoxArray.SetSize(0);

	m_bAppendBitmaps = FALSE;
	m_bAdornmentsAdded = FALSE;
	m_pWndCommandBar = this;
	m_hBCComboBox = NULL;
#if defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300)
// WinCE: PSPC 2.0 apps that want a "Shared New" button should set this to TRUE before calling Create(Ex).
	m_bShowSharedNewButton = FALSE;
	m_wNextMenuID = 0xF000;
#endif // _WIN32_WCE_PSPC
}

CCeCommandBar::~CCeCommandBar()
{
	RemoveCustomControls();
	if (m_hWnd != NULL)
	{
		CFrameWnd* pFrameWnd = GetParentFrame();
		if((pFrameWnd != NULL) && (pFrameWnd->m_hCommandBar == m_hWnd))
			pFrameWnd->m_hCommandBar = NULL;

		HWND hWnd = m_hWnd;
		Detach();
		::DestroyWindow(hWnd);
	}
}

#if defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300)
BEGIN_MESSAGE_MAP(CCeCommandBar, CToolBar)
	//{{AFX_MSG_MAP(CNotepadApp)
	ON_COMMAND(IDM_SHAREDNEW, CCeCommandBar::OnSharedNew)
	//}}AFX_MSG_MAP
END_MESSAGE_MAP()

void CCeCommandBar::OnSharedNew()
{
	CWinApp* pApp = AfxGetApp();
	ASSERT(pApp!=NULL);

	pApp->OnCmdMsg(ID_FILE_NEW, 0, NULL, NULL);		
}

BOOL CCeCommandBar::Create(CWnd* pParentWnd, DWORD dwStyle, UINT nID)
{
	return CreateEx(pParentWnd, 0, dwStyle,	CRect(0, 0, 0, 0), nID);
}

BOOL CCeCommandBar::CreateEx(CWnd* pParentWnd, DWORD dwCtrlStyle, DWORD dwStyle, 
							 CRect rcBorders, UINT nID)
{
	UINT nRsrcID = AFXCE_IDR_SCRATCH_SHMENU;
	HINSTANCE hInst = AfxFindResourceHandle(MAKEINTRESOURCE(nRsrcID), RT_RCDATA);
	ASSERT(hInst != NULL);

	// Create a scratch menu bar, which has the "Shared New" button
	SHMENUBARINFO mbi;
	memset(&mbi, 0, sizeof(SHMENUBARINFO));
	mbi.cbSize     = sizeof(SHMENUBARINFO);
	mbi.hwndParent = pParentWnd->GetSafeHwnd();
	mbi.nToolBarId = nRsrcID;
	mbi.hInstRes   = hInst; 

	if (!SHCreateMenuBar(&mbi)) 
	{
#ifdef _DEBUG
		TRACE1("Warning: SHCreateMenuBar failed: GetLastError returns 0x%8.8X\n",
			GetLastError());
#endif
	}

	if (mbi.hwndMB == NULL)
		return FALSE;

	VERIFY(SubclassWindow(mbi.hwndMB)); // Note: this is a worker window that is a parent of the toolbar.
	if (pParentWnd->IsFrameWnd())
	{
		m_pDockSite = (CFrameWnd*)pParentWnd;
		m_pDockSite->AddControlBar(this);
		m_pDockSite->m_hCommandBar = mbi.hwndMB;
	}

	if (!m_bShowSharedNewButton)
	{
		VERIFY(DefWindowProc(TB_DELETEBUTTON, 0, 0));
	}

	m_nCount = GetNumButtons();
	
	return TRUE;
}

// This function handles the new resource type used for PSPC 2.0 menu bars.
CMenu* CCeCommandBar::LoadSHMenuBar(LPCTSTR lpszResourceName)
{
	ASSERT_VALID(this);
	ASSERT(lpszResourceName != NULL);

	// This function requires that we called CCeCommandBar::Create(Ex) since the
	// parent window is established there.
	CWnd *pParentWnd = GetParent();
	ASSERT(pParentWnd != NULL);  

	// Clear out old commandbar if necessary.
	ResetCommandBar(); 
	DestroyWindow();
	ASSERT(m_hWnd == NULL);

	HINSTANCE hInst = AfxFindResourceHandle(lpszResourceName, RT_RCDATA);
	ASSERT(hInst != NULL);

	// Create the SHMenuBar
	SHMENUBARINFO mbi;
	memset(&mbi, 0, sizeof(SHMENUBARINFO));
	mbi.cbSize     = sizeof(SHMENUBARINFO);
	mbi.hwndParent = pParentWnd->GetSafeHwnd();
	mbi.nToolBarId = (int)lpszResourceName;
	mbi.hInstRes   = hInst; 

	if (!SHCreateMenuBar(&mbi)) 
	{
#ifdef _DEBUG
		TRACE1("Warning: SHCreateMenuBar failed: GetLastError returns 0x%8.8X\n",
			GetLastError());
#endif
	}

	if (mbi.hwndMB == NULL)
		return NULL;

	// Now subclass and mimic CControlBar::OnCreate().
	VERIFY(SubclassWindow(mbi.hwndMB)); 
	if (pParentWnd->IsFrameWnd())
	{
		m_pDockSite = (CFrameWnd*)pParentWnd;
		m_pDockSite->AddControlBar(this);
	}

	m_nCount = GetNumButtons();

	// Now get our handle to the menu by cracking the resource.
	HRSRC hRsrc = ::FindResource(hInst, lpszResourceName, RT_RCDATA);
	ASSERT(hRsrc != NULL);
	HGLOBAL hGlobal = ::LoadResource(hInst, hRsrc);
	ASSERT(hGlobal != NULL);
	short* pnMenuID = (short*)::LockResource(hGlobal);
	ASSERT(pnMenuID != NULL);
	short nMenuID = *pnMenuID;
	::UnlockResource(hGlobal);
	::FreeResource(hGlobal);

	if(nMenuID != 0)
	{	
		HMENU hMenu = ::LoadMenu(hInst, MAKEINTRESOURCE(nMenuID));
		ASSERT(hMenu != NULL);
		
		CMenu* pMenu = new CMenu;
		if(pMenu != NULL)
		{
			pMenu->Attach(hMenu);
			m_pMenuArray.Add(pMenu);
		}

		CFrameWnd* pFrameWnd = GetParentFrame();
		if(pFrameWnd != NULL)
		{
			// set the frame's m_hMenu so it can service CWnd::GetMenu()
			if(pFrameWnd->GetMenu() == NULL) 
				::WCE_FCTN(SetMenu)(pFrameWnd->m_hWnd, hMenu);
			pFrameWnd->DelayRecalcLayout(FALSE); 
		}

		return pMenu;
	}

	return NULL; // the SHMenuBar has been created, but we don't have a CMenu*.
}

// Since the PSPC 2.0 command bar is not inside the application frame window, it 
// doesn't consume any space in the layout calculations.
CSize CCeCommandBar::CalcFixedLayout(BOOL bStretch, BOOL bHorz)
{
	CSize sizeResult(0,0);
	return sizeResult;
}

CSize CCeCommandBar::CalcDynamicLayout(int nLength, DWORD nMode)
{
	CSize sizeResult(0,0);
	return sizeResult;
}
#endif // _WIN32_WCE_PSPC

void CCeCommandBar::RemoveCustomControls()
{
	CFrameWnd* pFrameWnd = GetParentFrame();

	int nLength = m_pMenuArray.GetSize();
	for(int n=0; n < nLength; n++)
	{
		CMenu* pMenu = (CMenu*)m_pMenuArray[n];
		ASSERT(pMenu);
	
		// unbind the menu from the frame window
		if((pFrameWnd != NULL) && (pFrameWnd->GetMenu() == pMenu))
			::WCE_FCTN(SetMenu)(pFrameWnd->m_hWnd, NULL);

		if(pMenu->m_hMenu != NULL)
			pMenu->DestroyMenu();
		delete pMenu;
	}

	nLength = m_pComboBoxArray.GetSize();
	for(n=0; n < nLength; n++)
	{
		CComboBox* pComboBox = (CComboBox*)m_pComboBoxArray[n];
		ASSERT(pComboBox);
		if(pComboBox->m_hWnd != NULL)
			pComboBox->Detach();
		delete pComboBox;
	}

	m_pMenuArray.SetSize(0);
	m_pComboBoxArray.SetSize(0);
}

void CCeCommandBar::ResetCommandBar()
{
	ASSERT_VALID(this);

	CWnd* pParentWnd = GetParent();
	DWORD dwPrevCtrlStyle = GetStyle();
	DWORD dwPrevTBStyle = m_dwStyle;
	CRect rcBorders(m_cxLeftBorder,m_cyTopBorder,m_cxRightBorder,m_cyBottomBorder);
	UINT nID = GetDlgCtrlID();

	ShowWindow(SW_HIDE);
	m_nCount = GetNumButtons();
 
	TBBUTTON button;
	while(m_nCount--)
	{
		_GetButton(0, &button);
		if(IsCustomControlButton(button))
			::DestroyWindow((HWND)button.dwData);
		VERIFY(DefWindowProc(TB_DELETEBUTTON, 0, 0));
	}

	RemoveCustomControls();
	DestroyWindow(); 

	AfxDeleteObject((HGDIOBJ*)&m_hbmImageWell);
	m_hRsrcImageWell = NULL;
	m_hInstImageWell = NULL;
	m_hbmImageWell = NULL;
	delete m_pStringMap;
	m_pStringMap = NULL;

	m_bAppendBitmaps = FALSE;
	m_bAdornmentsAdded = FALSE;
	m_pWndCommandBar = this;
	m_hBCComboBox = NULL;
#if defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300)
	m_wNextMenuID = 0xF000;
#endif // _WIN32_WCE_PSPC

	VERIFY(CreateEx(pParentWnd, dwPrevCtrlStyle, dwPrevTBStyle, rcBorders, nID));

	// Make sure all the styles are the same
	::SetWindowLong(m_hWnd, GWL_STYLE, dwPrevCtrlStyle);
	m_dwStyle = dwPrevTBStyle;

	CFrameWnd* pFrameWnd = GetParentFrame();
	if(pFrameWnd != NULL)
		pFrameWnd->DelayRecalcLayout(FALSE); 
}



CComboBox* CCeCommandBar::InsertComboBox(int nWidth, WORD idComboBox,
										 int nButton /*= CMDBAR_END*/, 
										 DWORD dwStyle /*= CBS_DROPDOWNLIST | WS_VSCROLL*/)
{
	return InsertComboBox(NULL, nWidth, idComboBox, nButton, dwStyle);
}


CComboBox* CCeCommandBar::InsertComboBox(CComboBox* pComboBox,
										 int nWidth, WORD idComboBox,
										 int nButton /*= CMDBAR_END*/, 
										 DWORD dwStyle /*= CBS_DROPDOWNLIST | WS_VSCROLL*/)
{
	ASSERT_VALID(this);
	ASSERT(::IsWindow(m_hWnd));
	ASSERT(nWidth >= 1);
	ASSERT((nButton == CMDBAR_END) || ((nButton >= 0) && (nButton <= GetNumButtons())));

	BOOL bTemporaryButton = FALSE;
	BOOL bSetBCCombo = FALSE;

	if(idComboBox == 0xFFFE)
	{
		// Backward compatibility combo box: no CComboBox allocated and can have at most one of
		// these combo boxes.
		ASSERT(m_hBCComboBox == NULL);
		idComboBox = IDC_CMDBAR_COMBOBOX;
		bSetBCCombo = TRUE;
	}

	if((nButton == CMDBAR_END) || (nButton == GetNumButtons()))
	{
		// WinCE: adding combo boxes at the end doesn't work quite right, 
		// so workaround with a temporary button.
		VERIFY(InsertSeparator(1));
		nButton = GetNumButtons()-1;
		bTemporaryButton = TRUE;
	}

	HWND hwndMB = ::GetWindow(m_hWnd, GW_CHILD);

	HWND hComboBox = ::CommandBar_InsertComboBox(hwndMB, AfxGetInstanceHandle(), nWidth, 
	                                             dwStyle | CBS_DROPDOWN, idComboBox, nButton);
	ASSERT(hComboBox != NULL);

	if(bTemporaryButton)
		VERIFY(DefWindowProc(TB_DELETEBUTTON, nButton+1, 0));

	if(bSetBCCombo)
	{
		m_hBCComboBox = hComboBox;
	}
	else if(pComboBox != NULL) // app is passing in the combo box
	{
		ASSERT(pComboBox->m_hWnd == NULL); // must be using a "fresh" CComboBox
		pComboBox->SubclassWindow(hComboBox);
	}
	else // CCeCommandBar needs to create a new CComboBox and own it
	{
		pComboBox = new CComboBox; 
		if(pComboBox != NULL)
		{
			pComboBox->SubclassWindow(hComboBox);
			m_pComboBoxArray.Add(pComboBox);
		}
		else
		{
			ASSERT(FALSE);
		}
	}

	m_nCount = GetNumButtons();
	SetToCustomControlButton(m_nCount-1);

	CFrameWnd* pFrame = GetParentFrame();
	if(pFrame != NULL)
		pFrame->DelayRecalcLayout(FALSE); 

	return (CComboBox*) pComboBox;
}

#if defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300)
void MenuItemStringStrip(HMENU hMenu);
#endif // _WIN32_WCE_PSPC

CMenu* CCeCommandBar::InsertMenuBar(LPCTSTR lpszMenuName, int nButton /*= CMDBAR_END*/)
{
	ASSERT_VALID(this);
	ASSERT(::IsWindow(m_hWnd));
	ASSERT(lpszMenuName != NULL);
	ASSERT((nButton == CMDBAR_END) || ((nButton >= 0) && (nButton <= GetNumButtons())));

	if(nButton == GetNumButtons())
		nButton = CMDBAR_END;

	HINSTANCE hInst = AfxFindResourceHandle(lpszMenuName, RT_MENU);
	ASSERT(hInst != NULL);

#if defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300)
	HMENU hMenu = ::LoadMenu(hInst, lpszMenuName);
	ASSERT((hMenu != NULL) && ::WCE_FCTN(IsMenu)(hMenu));

	TBBUTTON tbButton; 
	memset(&tbButton, 0, sizeof(TBBUTTON));
	tbButton.iBitmap = I_IMAGENONE;
	tbButton.fsState = TBSTATE_ENABLED;
	tbButton.fsStyle = TBSTYLE_DROPDOWN | TBSTYLE_NO_DROPDOWN_ARROW | TBSTYLE_AUTOSIZE;

	MENUITEMINFO mii;
	TCHAR szMenuItem[256];
	memset(&mii, 0, sizeof(MENUITEMINFO));
	mii.fMask = MIIM_TYPE;
	mii.fType = MFT_STRING;
	mii.dwTypeData = szMenuItem;
	
	int nIndex = 0;
	while(TRUE)
	{
		tbButton.dwData = (DWORD)::GetSubMenu(hMenu, nIndex);
		if(tbButton.dwData == NULL)
			break;

		mii.cch = 255;
		::GetMenuItemInfo(hMenu, nIndex, TRUE, &mii); // fills szMenuitem

		int n = 0;		
		for (int i =0; i < (int) mii.cch; i++) 
		{
			if( szMenuItem[i] == _T('&'))
				continue;
			szMenuItem[n++] = szMenuItem[i];
		}
		szMenuItem[n] = _T('\0');

		tbButton.iString = (int)szMenuItem;		
		tbButton.idCommand = m_wNextMenuID++;		
		VERIFY(DefWindowProc(TB_INSERTBUTTON, nButton, (LPARAM)&tbButton));
		MenuItemStringStrip((HMENU) tbButton.dwData );

		if(nButton != CMDBAR_END)
			nButton++;
		nIndex++;
	}

	m_nCount = GetNumButtons();
#else // _WIN32_WCE_PSPC
	BOOL bResult = ::CommandBar_InsertMenubarEx(m_hWnd, hInst, (LPTSTR)lpszMenuName, nButton);
	ASSERT(bResult);

	if(nButton == CMDBAR_END)
		nButton = GetNumButtons()-1;

	HMENU hMenu = ::CommandBar_GetMenu(m_hWnd, nButton);
	ASSERT((hMenu != NULL) && ::WCE_FCTN(IsMenu)(hMenu));

	m_nCount = GetNumButtons();
	SetToCustomControlButton(m_nCount-1);
#endif // _WIN32_WCE_PSPC

	CMenu* pMenu = new CMenu;
	if(pMenu != NULL)
	{
		pMenu->Attach(hMenu);
		m_pMenuArray.Add(pMenu);
	}

	CFrameWnd* pFrameWnd = GetParentFrame();
	if(pFrameWnd != NULL)
	{
		// set the frame's m_hMenu so it can service CWnd::GetMenu()
		if(pFrameWnd->GetMenu() == NULL) 
			::WCE_FCTN(SetMenu)(pFrameWnd->m_hWnd, hMenu);
		pFrameWnd->DelayRecalcLayout(FALSE); 
	}

	return pMenu;
}

CMenu* CCeCommandBar::GetMenuBar(int nButton /*= CMDBAR_END*/) const
{
	ASSERT_VALID(this);
	ASSERT(::IsWindow(m_hWnd));
	ASSERT((nButton >= 0) && (nButton < GetNumButtons()));

	HMENU hMenu = ::CommandBar_GetMenu(m_hWnd, nButton);
	if((hMenu == NULL) || !::WCE_FCTN(IsMenu)(hMenu))
		return NULL;

	// Find the CMenu for it.
	int nLength = m_pMenuArray.GetSize();
	for(int n=0; n < nLength; n++)
	{
		CMenu* pMenu = (CMenu*)m_pMenuArray[n];
		ASSERT(pMenu);
		if(pMenu->m_hMenu == hMenu)
			return pMenu;
	}

	return NULL;
}

BOOL CCeCommandBar::DrawMenuBar(int nButton) const
{
	ASSERT_VALID(this);
	ASSERT(::IsWindow(m_hWnd));
	ASSERT((nButton >= 0) && (nButton < GetNumButtons()));
#ifdef _DEBUG
	HMENU hMenu = ::CommandBar_GetMenu(m_hWnd, nButton);
	ASSERT((hMenu != NULL) && ::WCE_FCTN(IsMenu)(hMenu));
#endif

	return ::CommandBar_DrawMenuBar(m_hWnd, nButton);
}

BOOL CCeCommandBar::AddAdornments(DWORD dwFlags /* = 0 */)
{
#if !(defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300))
	ASSERT_VALID(this);
	ASSERT(::IsWindow(m_hWnd));
	ASSERT((dwFlags == 0) || (dwFlags & CMDBAR_OK) || (dwFlags & CMDBAR_HELP));
	ASSERT(!m_bAdornmentsAdded); // Can add adornments only once

	CFrameWnd* pFrame = GetParentFrame();

	if(pFrame != NULL)
		pFrame->RecalcLayout(FALSE); // This is necessary to right align the adornment buttons

	HIMAGELIST hImageList = (HIMAGELIST)DefWindowProc(TB_GETIMAGELIST,0,0);
	int nBitmapCount = ImageList_GetImageCount(hImageList);
	int nButtonCountBefore = GetNumButtons();

	if(!::CommandBar_AddAdornments(m_hWnd, dwFlags, 0))
		return FALSE;

	m_bAdornmentsAdded = TRUE;
	m_nCount = GetNumButtons();

	int nNumAdornments = 1;
	if(dwFlags & CMDBAR_OK)
		nNumAdornments++;
	if(dwFlags & CMDBAR_HELP)
		nNumAdornments++;

	// Adornments may have incorrect bitmap indices because of our ImageList_Add
	// workaround in bartool.cpp, so fix them.  The adornment indices are in decreasing
	// order.
	TBBUTTON button;
	for(int n = nButtonCountBefore; n < m_nCount; n++)
	{
		_GetButton(n, &button);
		if(button.fsStyle != TBSTYLE_SEP)
		{
			if((button.idCommand == WM_CLOSE) || (button.idCommand == IDOK))
			{
				button.iBitmap = nBitmapCount + nNumAdornments - 1;
				_SetButton(n, &button);
			}
			nNumAdornments--;
		}
	}

	// Since the adornment is right aligned, we should set a wrap here so new buttons 
	// go to the next line.
	WrapAtButton(m_nCount-1);
#endif // _WIN32_WCE_PSPC

	return TRUE;
}

BOOL CCeCommandBar::InsertSeparator(int nWidth /*= 6*/, int nButton /*= CMDBAR_END*/)
{
	ASSERT(nWidth > 0);
	ASSERT((nButton == CMDBAR_END) || ((nButton >= 0) && (nButton <= GetNumButtons())));

	if(nButton == GetNumButtons())
		nButton = CMDBAR_END;

	TBBUTTON separator = {nWidth, 0, TBSTATE_ENABLED, TBSTYLE_SEP, 0, 0, 0, 0};
	BOOL bReturn = DefWindowProc(TB_INSERTBUTTON, nButton, (LPARAM)&separator);
	m_nCount = GetNumButtons();

	CFrameWnd* pFrame = GetParentFrame();
	if(pFrame != NULL)
		pFrame->DelayRecalcLayout(FALSE); 

	return bReturn;
}

void CCeCommandBar::WrapAtButton(int nButton)
{
	ASSERT((nButton >= 0) && (nButton < GetNumButtons()));

	// You can't wrap on custom controls because the width of the button (a separator) is 
	// interpreted as the separator height, so add a small separator instead.
	TBBUTTON button;
	_GetButton(nButton, &button);
	if(IsCustomControlButton(button))
	{
		nButton++;
		InsertSeparator(1, nButton);
	}

	SetButtonStyle(nButton, GetButtonStyle(nButton) | TBBS_WRAPPED);

	CFrameWnd* pFrame = GetParentFrame();
	if(pFrame != NULL)
		pFrame->DelayRecalcLayout(FALSE); 
}

// MFC expects custom controls to have their idcommand set to -1.
void CCeCommandBar::SetToCustomControlButton(int nButton)
{
	ASSERT((nButton == CMDBAR_END) || ((nButton >= 0) && (nButton < GetNumButtons())));

	TBBUTTON button;
	_GetButton(nButton, &button);
	button.idCommand = -1;
	_SetButton(nButton, &button);
}

// The following is used to test for menus and combo boxes (and other custom controls)
// Note: we don't use the (idCommand == -1) property because that's not guaranteed in 
// general.
BOOL CCeCommandBar::IsCustomControlButton(const TBBUTTON& button)
{
	return ( (button.fsStyle == TBSTYLE_SEP) && 
		     (button.dwData != NULL)         && 
			 ::IsWindow((HWND)button.dwData) );
}


#if defined(_WIN32_WCE_PSPC) && (_WIN32_WCE >= 300)
void MenuItemStringStrip(HMENU hMenu)
{
	MENUITEMINFO mii;
	TCHAR szMenuItem[256];

	int nIndex = 0;
	while(TRUE)
	{
		memset(&mii, 0, sizeof(MENUITEMINFO));
		mii.fMask = MIIM_TYPE;
		mii.dwTypeData = szMenuItem;

		mii.cch = 255;
		BOOL bResult = ::GetMenuItemInfo(hMenu, nIndex, TRUE, &mii); // fills szMenuitem
		if (!bResult) break;

		if ( mii.fType == MFT_STRING) {
			int m = 0;
			for ( int i =0; i < (int) mii.cch; i++) 
			{
				if( szMenuItem[i] == _T('&'))
					continue;
				if( szMenuItem[i] == _T('\t'))
					break;
				szMenuItem[m++] = szMenuItem[i];
			}

			szMenuItem[m] = _T('\0');		
			mii.cch = m;
			mii.fMask = MIIM_TYPE;
			::SetMenuItemInfo(hMenu, nIndex, TRUE, &mii);
		}		
		nIndex++;
	}
}
#endif // _WIN32_WCE_PSPC

#endif // _WIN32_WCE_NO_CONTROLBARS
/////////////////////////////////////////////////////////////////////////////

#ifdef AFX_INIT_SEG
#pragma code_seg(AFX_INIT_SEG)
#endif

#if !defined(_WIN32_WCE_NO_CONTROLBARS)
IMPLEMENT_DYNAMIC(CCeCommandBar, CToolBar)
#endif // _WIN32_WCE_NO_CONTROLBARS
